<?php
/*
Plugin Name: Neat Affiliate Links
Description: Lightweight (2,4 kB) affiliate links plugin with categories, customizable slug, nofollow, 307 redirects, and click tracking.
Version: 1.3
Author: NeatWP
Author URI: https://neatwp.com/
*/

if (!defined('ABSPATH')) {
    exit;
}

function nal_register_affiliate_links() {
    $labels = array(
        'name' => 'Affiliate Links',
        'singular_name' => 'Affiliate Link',
        'menu_name' => 'Affiliate Links',
        'add_new' => 'Add New',
        'add_new_item' => 'Add New Affiliate Link',
        'edit_item' => 'Edit Affiliate Link',
        'new_item' => 'New Affiliate Link',
        'view_item' => 'View Affiliate Link',
        'search_items' => 'Search Affiliate Links',
        'not_found' => 'No affiliate links found',
        'not_found_in_trash' => 'No affiliate links found in Trash',
    );

    // Get custom slug, default to 'out'
    $custom_slug = get_option('nal_affiliate_slug', 'out');

    $args = array(
        'labels' => $labels,
        'public' => true,
        'publicly_queryable' => true,
        'show_ui' => true,
        'show_in_menu' => true,
        'query_var' => true,
        'rewrite' => array('slug' => $custom_slug, 'with_front' => false),
        'capability_type' => 'post',
        'has_archive' => false,
        'hierarchical' => false,
        'menu_position' => 100,
        'supports' => array('title'),
        'taxonomies' => array('affiliate_category'),
    );

    register_post_type('affiliate_link', $args);

    $cat_labels = array(
        'name' => 'Affiliate Categories',
        'singular_name' => 'Affiliate Category',
        'search_items' => 'Search Categories',
        'all_items' => 'All Categories',
        'parent_item' => 'Parent Category',
        'parent_item_colon' => 'Parent Category:',
        'edit_item' => 'Edit Category',
        'update_item' => 'Update Category',
        'add_new_item' => 'Add New Category',
        'new_item_name' => 'New Category Name',
        'menu_name' => 'Categories',
    );

    $cat_args = array(
        'hierarchical' => true,
        'labels' => $cat_labels,
        'show_ui' => true,
        'show_admin_column' => true,
        'public' => false,
        'publicly_queryable' => false,
        'query_var' => false,
        'rewrite' => false,
    );

    register_taxonomy('affiliate_category', array('affiliate_link'), $cat_args);
}
add_action('init', 'nal_register_affiliate_links');

function nal_add_settings_page() {
    add_submenu_page(
        'edit.php?post_type=affiliate_link',
        'Affiliate Links Settings',
        'Settings',
        'manage_options',
        'nal-settings',
        'nal_settings_page_callback'
    );
}
add_action('admin_menu', 'nal_add_settings_page');

function nal_settings_page_callback() {
    ?>
    <div class="wrap">
        <h1>Neat Affiliate Links Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('nal_settings_group');
            do_settings_sections('nal-settings');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

function nal_register_settings() {
    register_setting('nal_settings_group', 'nal_affiliate_slug', array(
        'sanitize_callback' => 'nal_sanitize_slug',
        'default' => 'out'
    ));

    add_settings_section(
        'nal_main_section',
        'Link Settings',
        null,
        'nal-settings'
    );

    add_settings_field(
        'nal_affiliate_slug',
        'Affiliate Link Slug',
        'nal_affiliate_slug_callback',
        'nal-settings',
        'nal_main_section'
    );
}
add_action('admin_init', 'nal_register_settings');

function nal_sanitize_slug($input) {
    // Sanitize and lowercase the slug, remove invalid characters
    $input = sanitize_title($input);
    return empty($input) ? 'out' : $input;
}

function nal_affiliate_slug_callback() {
    $slug = get_option('nal_affiliate_slug', 'out');
    ?>
    <input type="text" name="nal_affiliate_slug" value="<?php echo esc_attr($slug); ?>" />
    <p class="description">Enter the slug to use for affiliate links (e.g., 'out' will make links like yoursite.com/out/link-name). Use lowercase letters, numbers, and hyphens only.</p>
    <p class="description"><strong>Note:</strong> After changing the slug, go to Settings > Permalinks and click "Save Changes" to flush rewrite rules.</p>
    <?php
}

function nal_add_meta_box() {
    add_meta_box(
        'nal_url_meta_box',
        'Affiliate Link Details',
        'nal_url_meta_box_callback',
        'affiliate_link',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'nal_add_meta_box');

function nal_url_meta_box_callback($post) {
    wp_nonce_field('nal_save_meta_box_data', 'nal_meta_box_nonce');
    $affiliate_url = get_post_meta($post->ID, '_nal_affiliate_url', true);
    $click_count = get_post_meta($post->ID, '_nal_click_count', true) ? (int) get_post_meta($post->ID, '_nal_click_count', true) : 0;
    ?>
    <p>
        <label for="nal_affiliate_url">Destination URL:</label><br>
        <input type="url" id="nal_affiliate_url" name="nal_affiliate_url" value="<?php echo esc_url($affiliate_url); ?>" style="width: 100%;" />
    </p>
    <p>
        <label>Click Count:</label><br>
        <strong><?php echo esc_html($click_count); ?> clicks</strong>
    </p>
    <?php
}

// Save Meta Box Data
function nal_save_meta_box_data($post_id) {
    if (!isset($_POST['nal_meta_box_nonce']) || !wp_verify_nonce($_POST['nal_meta_box_nonce'], 'nal_save_meta_box_data')) {
        return;
    }

    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    if (isset($_POST['nal_affiliate_url'])) {
        update_post_meta($post_id, '_nal_affiliate_url', esc_url_raw($_POST['nal_affiliate_url']));
    }
}
add_action('save_post', 'nal_save_meta_box_data');

function nal_add_click_count_column($columns) {
    $columns['click_count'] = 'Clicks';
    return $columns;
}
add_filter('manage_affiliate_link_posts_columns', 'nal_add_click_count_column');

function nal_display_click_count_column($column, $post_id) {
    if ($column === 'click_count') {
        $click_count = get_post_meta($post_id, '_nal_click_count', true) ? (int) get_post_meta($post_id, '_nal_click_count', true) : 0;
        echo esc_html($click_count);
    }
}
add_action('manage_affiliate_link_posts_custom_column', 'nal_display_click_count_column', 10, 2);

// Handle Redirects, Headers, and Click Tracking
function nal_template_redirect() {
    if (is_singular('affiliate_link')) {
        $post_id = get_the_ID();
        $affiliate_url = get_post_meta($post_id, '_nal_affiliate_url', true);
        if ($affiliate_url) {
            
            $click_count = get_post_meta($post_id, '_nal_click_count', true) ? (int) get_post_meta($post_id, '_nal_click_count', true) : 0;
            update_post_meta($post_id, '_nal_click_count', $click_count + 1);
            
            
            header('X-Robots-Tag: noindex, nofollow', true);
            
            wp_redirect($affiliate_url, 307);
            exit;
        } else {
            wp_redirect(home_url());
            exit;
        }
    }
}
add_action('template_redirect', 'nal_template_redirect');

add_action('template_redirect', 'nal_template_redirect');

// Exclude affiliate_link post type from WordPress sitemap
function nal_exclude_affiliate_link_from_sitemap($post_types) {
    unset($post_types['affiliate_link']);
    return $post_types;
}
add_filter('wp_sitemaps_post_types', 'nal_exclude_affiliate_link_from_sitemap');
